
#include <stdio.h>
#include <strings.h>
#include <stdlib.h>
#include <math.h>

#include <exec/types.h>

#include <proto/exec.h>
#include <proto/dos.h>

#include <proto/ahi.h>

#include <devices/ahi.h>
#include "os3_compat.h"

#include "hvl_replay.h"

uint32          audiobuflen;
int8           *audiobuffer[2]  = { NULL, NULL };

// Libraries and interfaces
struct Library  *AHIBase = NULL;

// AHI stuff
struct MsgPort    *ahi_mp;
struct AHIRequest *ahi_io[2] = { NULL, NULL };
int32              ahi_dev = -1;
uint32             ahisig = 0;
struct AHIRequest *prev_req = NULL;

BOOL   need_wait[2];
uint32 nextbuf;

#define FREQ 32000

BOOL init( void )
{
  uint32 i;

  audiobuflen = FREQ * sizeof( uint16 ) * 2 / 50;

  audiobuffer[0] = AllocVec( audiobuflen * 2, MEMF_ANY );
  if( audiobuffer[0] == NULL )
  {
    printf( "Out of memory!\n" );
    return FALSE;
  }

  audiobuffer[1] = &audiobuffer[0][audiobuflen];
  
  for( i=0; i<audiobuflen; i++ )
    audiobuffer[0][i] = 0;  

  ahi_mp = CreateMsgPort();
  if( !ahi_mp )
  {
    printf( "Unable to create message port!\n" );
    return FALSE;
  }

  ahi_io[0] = (struct AHIRequest *)CreateIORequest( ahi_mp, sizeof( struct AHIRequest ) );
  if( ahi_io[0] == NULL ) return FALSE;

  ahi_io[0]->ahir_Version = 4;
  
  ahi_dev = OpenDevice( AHINAME, AHI_DEFAULT_UNIT, (struct IORequest *)ahi_io[0], 0 ); 
  if( ahi_dev == -1 ) return FALSE;
  
  AHIBase = (struct Library *)ahi_io[0]->ahir_Std.io_Device;
  
  ahi_io[1] = AllocVec( sizeof( struct AHIRequest ), MEMF_ANY );
  if( ahi_io[1] == NULL ) return FALSE;
  
  CopyMem( ahi_io[0], ahi_io[1], sizeof( struct AHIRequest ) );

  ahisig  = 1L<<ahi_mp->mp_SigBit;
  
  return TRUE;
}

void shut( void )
{
  if( ahi_io[1] )     FreeVec( ahi_io[1] );
  if( ahi_dev != -1 )
  {
    CloseDevice( (struct IORequest *)ahi_io[0] );
    DeleteIORequest( (struct IORequest *)ahi_io[0] );
  }
  if( ahi_mp )        DeleteMsgPort( ahi_mp );
  if( audiobuffer[0] ) FreeVec( audiobuffer[0] );
}

void mix_and_play( struct hvl_tune *ht )
{
 
  if( need_wait[nextbuf] )
    WaitIO( (struct IORequest *)ahi_io[nextbuf] );

  hvl_DecodeFrame( ht, audiobuffer[nextbuf], audiobuffer[nextbuf]+sizeof( int16 ), 4 );

  ahi_io[nextbuf]->ahir_Std.io_Command = CMD_WRITE;
  ahi_io[nextbuf]->ahir_Std.io_Data    = audiobuffer[nextbuf];
  ahi_io[nextbuf]->ahir_Std.io_Length  = audiobuflen;
  ahi_io[nextbuf]->ahir_Std.io_Offset  = 0;
  ahi_io[nextbuf]->ahir_Type           = AHIST_S16S;
  ahi_io[nextbuf]->ahir_Frequency      = FREQ;
  ahi_io[nextbuf]->ahir_Volume         = 0x10000;
  ahi_io[nextbuf]->ahir_Position       = 0x8000;
  ahi_io[nextbuf]->ahir_Link           = prev_req;
  SendIO( (struct IORequest *)ahi_io[nextbuf] );
            
  prev_req = ahi_io[nextbuf];
  need_wait[nextbuf] = TRUE;

  nextbuf ^= 1;  
  
  if( ( need_wait[nextbuf] == TRUE ) &&
      ( CheckIO( (struct IORequest *)ahi_io[nextbuf] ) != NULL ) )
  {
    WaitIO( (struct IORequest *)ahi_io[nextbuf] );
    need_wait[nextbuf] = FALSE;
  }
            
  if( need_wait[nextbuf] == FALSE )
  {
    hvl_DecodeFrame( ht, audiobuffer[nextbuf], audiobuffer[nextbuf]+sizeof( int16 ), 4 );

    ahi_io[nextbuf]->ahir_Std.io_Command = CMD_WRITE;
    ahi_io[nextbuf]->ahir_Std.io_Data    = audiobuffer[nextbuf];
    ahi_io[nextbuf]->ahir_Std.io_Length  = audiobuflen;
    ahi_io[nextbuf]->ahir_Std.io_Offset  = 0;
    ahi_io[nextbuf]->ahir_Type           = AHIST_S16S;
    ahi_io[nextbuf]->ahir_Frequency      = FREQ;
    ahi_io[nextbuf]->ahir_Volume         = 0x10000;
    ahi_io[nextbuf]->ahir_Position       = 0x8000;
    ahi_io[nextbuf]->ahir_Link           = prev_req;
    SendIO( (struct IORequest *)ahi_io[nextbuf] );

    prev_req = ahi_io[nextbuf];
    need_wait[nextbuf] = TRUE;
    nextbuf ^= 1;  
  }
}

int main( int argc, char *argv[] )
{
  struct hvl_tune *tune;

  if( argc < 2 )
  {
    printf( "Usage: play_hvl <tune>\n" );
    return 0;
  }

  if( init() )
  {
    hvl_InitReplayer();
    tune = hvl_LoadTune( argv[1], FREQ, 2 );
    if( tune )
    {
      BOOL done;
      uint32 gotsigs;
      
      hvl_InitSubsong( tune, 0 );

      nextbuf = 0;
      mix_and_play( tune );

      done = FALSE;
      
      while( !done )
      {
        gotsigs = Wait( ahisig | SIGBREAKF_CTRL_C );
      
        if( gotsigs & ahisig ) mix_and_play( tune );
        if( gotsigs & SIGBREAKF_CTRL_C ) done = TRUE;
      }
      hvl_FreeTune( tune );
    }
  }
  shut();
  return 0;
}
